import type { Plugin as VitePlugin } from 'vite';
interface ThemeJsonPluginOptions {
    /**
     * Path to the Tailwind configuration file.
     * This is used as a source of truth for generating theme.json settings.
     * If not provided, only CSS variables from the @theme block will be processed.
     *
     * @example './tailwind.config.js'
     */
    tailwindConfig?: string;
    /**
     * Labels for color shades to use in the WordPress editor.
     * Keys should be shade numbers (e.g. 50, 100, 500) and values are the human-readable labels.
     * For example: { 50: 'Lightest', 100: 'Lighter', 500: 'Default' }
     * When provided, color names will be formatted as '{label} {color}' instead of '{color} ({shade})'.
     */
    shadeLabels?: Record<string, string>;
    /**
     * Whether to disable generating color palette entries in theme.json.
     * When true, no color variables will be processed from the @theme block.
     *
     * @default false
     */
    disableTailwindColors?: boolean;
    /**
     * Whether to disable generating font family entries in theme.json.
     * When true, no font-family variables will be processed from the @theme block.
     *
     * @default false
     */
    disableTailwindFonts?: boolean;
    /**
     * Whether to disable generating font size entries in theme.json.
     * When true, no font-size variables will be processed from the @theme block.
     *
     * @default false
     */
    disableTailwindFontSizes?: boolean;
}
/**
 * Supported file extensions for WordPress imports transformation
 */
declare const SUPPORTED_EXTENSIONS: readonly [".js", ".jsx", ".ts", ".tsx"];
type SupportedExtension = (typeof SUPPORTED_EXTENSIONS)[number];
/**
 * Configuration for the WordPress plugin
 */
interface WordPressPluginConfig {
    /**
     * File extensions to process for WordPress imports.
     *
     * @default ['.js', '.jsx', '.ts', '.tsx']
     */
    extensions?: SupportedExtension[];
    /**
     * HMR configuration for the WordPress editor
     */
    hmr?: {
        /**
         * Pattern to match editor entry points.
         * Can be a string (exact match) or RegExp.
         *
         * @default /editor/
         */
        editorPattern?: string | RegExp;
        /**
         * Pattern to match editor CSS files.
         * Can be a string (exact match) or RegExp.
         *
         * @default 'editor.css'
         */
        cssPattern?: string | RegExp;
        /**
         * Whether to enable HMR for the WordPress editor.
         *
         * @default true
         */
        enabled?: boolean;
        /**
         * Name of the editor iframe element.
         *
         * @default 'editor-canvas'
         */
        iframeName?: string;
    };
}
/**
 * Creates a Vite plugin that handles WordPress dependencies.
 * This plugin transforms @wordpress/* imports into global wp.* references,
 * generates a dependency manifest for WordPress enqueuing, and handles
 * external dependencies.
 *
 * @example
 * ```ts
 * // vite.config.ts
 * import { wordpressPlugin } from '@wordpress/vite-plugin'
 *
 * export default defineConfig({
 *   plugins: [
 *     wordpressPlugin()
 *   ]
 * })
 * ```
 *
 * The plugin will:
 * 1. Transform imports like `import { useState } from '@wordpress/element'`
 *    into `const useState = wp.element.useState`
 * 2. Track WordPress script dependencies (e.g. 'wp-element')
 * 3. Generate an editor.deps.json file listing all dependencies
 * 4. Mark all @wordpress/* packages as external dependencies
 * 5. Prevent WordPress core libraries from being bundled
 *
 * @returns A Vite plugin configured to handle WordPress dependencies
 */
export declare function wordpressPlugin(config?: WordPressPluginConfig): VitePlugin;
/**
 * Configuration for the WordPress theme.json plugin
 */
interface ThemeJsonConfig extends ThemeJsonPluginOptions {
    /**
     * The Tailwind configuration object containing design tokens.
     * This is used as a source of truth for generating theme.json settings.
     * If not provided, only CSS variables from the @theme block will be processed.
     */
    tailwindConfig?: string;
    /**
     * The path to the base theme.json file.
     *
     * @default './theme.json'
     */
    baseThemeJsonPath?: string;
    /**
     * The path where the generated theme.json will be written.
     *
     * @default 'assets/theme.json'
     */
    outputPath?: string;
    /**
     * The CSS file to process for theme variables.
     *
     * @default 'app.css'
     */
    cssFile?: string;
}
/**
 * Creates a Vite plugin that generates a WordPress theme.json file from Tailwind CSS variables.
 * This allows theme.json settings to stay in sync with your Tailwind design tokens.
 *
 * @example
 * ```ts
 * // vite.config.ts
 * import { wordpressThemeJson } from '@wordpress/vite-plugin'
 * import tailwindConfig from './tailwind.config.js'
 *
 * export default defineConfig({
 *   plugins: [
 *     wordpressThemeJson({
 *       disableTailwindColors: false,
 *       disableTailwindFonts: false,
 *       disableTailwindFontSizes: false,
 *     }),
 *   ]
 * })
 * ```
 *
 * CSS variables in an @theme block will be transformed into theme.json:
 * ```css
 * @theme {
 *   --color-primary: #000000;  ->  { name: "primary", color: "#000000" }
 *   --color-red-500: #ef4444;  ->  { name: "red-500", color: "#ef4444" }
 *   --font-inter: "Inter";       ->  { name: "inter", fontFamily: "Inter" }
 *   --text-lg: 1.125rem;         ->  { name: "lg", size: "1.125rem" }
 * }
 * ```
 *
 * @param options - Configuration options for the theme.json generator
 * @returns A Vite plugin configured to generate theme.json from CSS variables
 */
export declare function wordpressThemeJson(config?: ThemeJsonConfig): VitePlugin;
export {};
