import {
  defaultRequestToExternal,
  defaultRequestToHandle
} from "@wordpress/dependency-extraction-webpack-plugin/lib/util.js";
import fs from "fs";
import path from "path";
const SUPPORTED_EXTENSIONS = [".js", ".jsx", ".ts", ".tsx"];
function wordpressPlugin(config = {}) {
  const extensions = config.extensions ?? SUPPORTED_EXTENSIONS;
  const dependencies = /* @__PURE__ */ new Set();
  const hmrConfig = {
    enabled: true,
    editorPattern: /editor/,
    cssPattern: "editor.css",
    iframeName: "editor-canvas",
    ...config.hmr
  };
  const hmrCode = `
if (import.meta.hot) {
    import.meta.hot.on('vite:beforeUpdate', (payload) => {
        const cssUpdates = payload.updates.filter(update => update.type === 'css-update');

        if (cssUpdates.length > 0) {
            const update = cssUpdates[0];

            // Find the iframe
            const editorIframe = document.querySelector('iframe[name="${hmrConfig.iframeName}"]');
            if (!editorIframe?.contentDocument) {
                window.location.reload();
                return;
            }

            // Find the existing style tag in the iframe
            const styles = editorIframe.contentDocument.getElementsByTagName('style');
            let editorStyle = null;
            for (const style of styles) {
                if (style.textContent.includes('${hmrConfig.cssPattern}')) {
                    editorStyle = style;
                    break;
                }
            }

            if (!editorStyle) {
                window.location.reload();
                return;
            }

            // Update the style content with new import and cache-busting timestamp
            const timestamp = Date.now();
            editorStyle.textContent = \`@import url('\${window.__vite_client_url}\${update.path}?t=\${timestamp}')\`;
            return;
        }

        // For non-CSS updates, reload
        window.location.reload();
    });
}`;
  function extractNamedImports(imports) {
    return imports.match(/{([^}]+)}/)?.at(1)?.split(",").map((s) => s.trim()).filter(Boolean) ?? [];
  }
  function handleNamedReplacement(namedImports, external) {
    const externalPath = external.join(".");
    return namedImports.map((importStr) => {
      const parts = importStr.split(" as ").map((s) => s.trim());
      const name = parts[0];
      const alias = parts[1] ?? name;
      return `const ${alias} = ${externalPath}.${name};`;
    }).join("\n");
  }
  return {
    name: "wordpress-plugin",
    enforce: "pre",
    options(opts) {
      return {
        ...opts,
        external: (id) => typeof id === "string" && id.startsWith("@wordpress/")
      };
    },
    resolveId(id) {
      if (!id?.startsWith("@wordpress/"))
        return null;
      const [external, handle] = [
        defaultRequestToExternal(id),
        defaultRequestToHandle(id)
      ];
      if (!external || !handle)
        return null;
      dependencies.add(handle);
      return { id, external: true };
    },
    transform(code, id) {
      if (!extensions.some((ext) => id.endsWith(ext)))
        return null;
      let transformedCode = code;
      const importRegex = /^[\s\n]*import[\s\n]+(?:([^;'"]+?)[\s\n]+from[\s\n]+)?['"]@wordpress\/([^'"]+)['"][\s\n]*;?/gm;
      let match;
      while ((match = importRegex.exec(code)) !== null) {
        const [fullMatch, imports, pkg] = match;
        const external = defaultRequestToExternal(`@wordpress/${pkg}`);
        const handle = defaultRequestToHandle(`@wordpress/${pkg}`);
        if (!external || !handle)
          continue;
        dependencies.add(handle);
        if (!imports) {
          transformedCode = transformedCode.replace(fullMatch, "");
          continue;
        }
        let replacement;
        if (imports.includes("{")) {
          replacement = handleNamedReplacement(
            extractNamedImports(imports),
            external
          );
        } else if (imports.includes("*")) {
          const namespaceAlias = imports.match(/\*\s+as\s+(\w+)/)?.[1];
          if (namespaceAlias) {
            replacement = `const ${namespaceAlias} = ${external.join(
              "."
            )};`;
          }
        } else {
          const defaultImport = imports.match(/^(\w+)/)?.[1];
          if (defaultImport) {
            replacement = `const ${defaultImport} = ${external.join(
              "."
            )};`;
          }
        }
        if (replacement) {
          transformedCode = transformedCode.replace(
            fullMatch,
            replacement
          );
        }
      }
      if (hmrConfig.enabled && !transformedCode.includes("vite:beforeUpdate") && (typeof hmrConfig.editorPattern === "string" && id.includes(hmrConfig.editorPattern) || hmrConfig.editorPattern instanceof RegExp && hmrConfig.editorPattern.test(id))) {
        transformedCode = `${transformedCode}
${hmrCode}`;
      }
      return {
        code: transformedCode,
        map: null
      };
    },
    generateBundle() {
      this.emitFile({
        type: "asset",
        name: "editor.deps.json",
        originalFileName: "editor.deps.json",
        source: JSON.stringify([...dependencies], null, 2)
      });
    }
  };
}
function mergeThemeWithExtend(theme) {
  if (!theme.extend)
    return theme;
  return {
    ...theme,
    colors: {
      ...theme.colors,
      ...theme.extend.colors
    },
    fontFamily: {
      ...theme.fontFamily,
      ...theme.extend.fontFamily
    },
    fontSize: {
      ...theme.fontSize,
      ...theme.extend.fontSize
    }
  };
}
function flattenColors(colors) {
  const flattened = [];
  for (const [name, value] of Object.entries(colors)) {
    if (typeof value === "string") {
      flattened.push([name, value]);
    } else if (typeof value === "object" && value !== null) {
      for (const [shade, shadeValue] of Object.entries(value)) {
        if (typeof shadeValue === "string") {
          flattened.push([`${name}-${shade}`, shadeValue]);
        }
      }
    }
  }
  return flattened;
}
function processFontFamilies(fonts) {
  return Object.entries(fonts).map(([name, value]) => {
    const fontFamily = Array.isArray(value) ? value.join(", ") : value;
    return {
      name,
      slug: name.toLowerCase(),
      fontFamily
    };
  });
}
function processFontSizes(sizes) {
  return Object.entries(sizes).map(([name, value]) => {
    const size = Array.isArray(value) ? value[0] : value;
    return {
      name,
      slug: name.toLowerCase(),
      size
    };
  });
}
async function loadTailwindConfig(configPath) {
  try {
    const absolutePath = path.resolve(configPath);
    const config = await import(absolutePath);
    const resolvedConfig = config.default || config;
    if (resolvedConfig.theme?.extend) {
      resolvedConfig.theme = mergeThemeWithExtend(resolvedConfig.theme);
    }
    return resolvedConfig;
  } catch (error) {
    throw new Error(
      `Failed to load Tailwind config from ${configPath}: ${error instanceof Error ? error.message : String(error)}`
    );
  }
}
function wordpressThemeJson(config = {}) {
  const {
    tailwindConfig,
    disableTailwindColors = false,
    disableTailwindFonts = false,
    disableTailwindFontSizes = false,
    baseThemeJsonPath = "./theme.json",
    outputPath = "assets/theme.json",
    cssFile = "app.css"
  } = config;
  let cssContent = null;
  let resolvedTailwindConfig;
  if (tailwindConfig !== void 0 && typeof tailwindConfig !== "string") {
    throw new Error("tailwindConfig must be a string path or undefined");
  }
  function extractThemeContent(css) {
    const themeMatch = css.match(/@(?:layer\s+)?theme\s*{/s);
    if (!themeMatch?.index)
      return null;
    const startIndex = themeMatch.index + themeMatch[0].length;
    const tokens = {
      ESCAPE: { pattern: "\\", skip: 1 },
      STRING: { pattern: /['"]/, handleUntil: (quote) => quote },
      COMMENT: { pattern: "/*", handleUntil: "*/" },
      OPEN_BRACE: { pattern: "{", count: 1 },
      CLOSE_BRACE: { pattern: "}", count: -1 }
    };
    let braceCount = 1;
    let position = startIndex;
    while (position < css.length) {
      if (css[position] === tokens.ESCAPE.pattern) {
        position += tokens.ESCAPE.skip + 1;
        continue;
      }
      if (/['"]/.test(css[position])) {
        const quote = css[position];
        position++;
        while (position < css.length) {
          if (css[position] === tokens.ESCAPE.pattern) {
            position += tokens.ESCAPE.skip + 1;
          } else if (css[position] === quote) {
            position++;
            break;
          } else {
            position++;
          }
        }
        continue;
      }
      if (css.slice(position, position + 2) === "/*") {
        position += 2;
        while (position < css.length) {
          if (css.slice(position, position + 2) === "*/") {
            position += 2;
            break;
          }
          position++;
        }
        continue;
      }
      if (css[position] === "{")
        braceCount++;
      if (css[position] === "}")
        braceCount--;
      if (braceCount === 0) {
        return css.substring(startIndex, position);
      }
      position++;
    }
    const blockType = themeMatch[0].trim();
    throw new Error(`Unclosed ${blockType} block - missing closing brace`);
  }
  return {
    name: "wordpress-theme-json",
    enforce: "pre",
    async configResolved() {
      if (tailwindConfig) {
        resolvedTailwindConfig = await loadTailwindConfig(
          tailwindConfig
        );
      }
    },
    transform(code, id) {
      if (id.includes(cssFile)) {
        cssContent = code;
      }
      return null;
    },
    async generateBundle() {
      try {
        const baseThemeJson = JSON.parse(
          fs.readFileSync(path.resolve(baseThemeJsonPath), "utf8")
        );
        const themeContent = cssContent ? extractThemeContent(cssContent) : null;
        if (!themeContent && !resolvedTailwindConfig)
          return;
        const extractVariables = (regex, content) => {
          if (!content)
            return [];
          const variables = [];
          let match;
          while ((match = regex.exec(content)) !== null) {
            const [, name, value] = match;
            if (name && value)
              variables.push([name, value.trim()]);
          }
          return variables;
        };
        const patterns = {
          COLOR: /--color-([^:]+):\s*([^;}]+)[;}]?/g,
          FONT_FAMILY: /--font-([^:]+):\s*([^;}]+)[;}]?/g,
          FONT_SIZE: /--text-([^:]+):\s*([^;}]+)[;}]?/g
        };
        const colorEntries = !disableTailwindColors ? [
          ...extractVariables(patterns.COLOR, themeContent).filter(([name]) => !name.endsWith("-*")).map(([name, value]) => {
            const parts = name.split("-");
            const colorName = parts[0];
            const shade = parts.length > 1 ? parts.slice(1).join(" ") : void 0;
            const capitalizedColor = colorName.charAt(0).toUpperCase() + colorName.slice(1);
            const displayName = shade ? config.shadeLabels && shade in config.shadeLabels ? `${config.shadeLabels[shade]} ${capitalizedColor}` : Number.isNaN(Number(shade)) ? `${capitalizedColor} (${shade.split(" ").map(
              (word) => word.charAt(0).toUpperCase() + word.slice(1)
            ).join(" ")})` : `${capitalizedColor} (${shade})` : capitalizedColor;
            return {
              name: displayName,
              slug: name.toLowerCase(),
              color: value
            };
          }),
          ...resolvedTailwindConfig?.theme?.colors ? flattenColors(
            resolvedTailwindConfig.theme.colors
          ).map(([name, value]) => {
            const parts = name.split("-");
            const colorName = parts[0];
            const shade = parts.length > 1 ? parts.slice(1).join(" ") : void 0;
            const capitalizedColor = colorName.charAt(0).toUpperCase() + colorName.slice(1);
            const displayName = shade ? config.shadeLabels && shade in config.shadeLabels ? `${config.shadeLabels[shade]} ${capitalizedColor}` : Number.isNaN(Number(shade)) ? `${capitalizedColor} (${shade.split(" ").map(
              (word) => word.charAt(0).toUpperCase() + word.slice(1)
            ).join(" ")})` : `${capitalizedColor} (${shade})` : capitalizedColor;
            return {
              name: displayName,
              slug: name.toLowerCase(),
              color: value
            };
          }) : []
        ] : void 0;
        const invalidFontProps = [
          "feature-settings",
          "variation-settings",
          "family",
          "size",
          "smoothing",
          "style",
          "weight",
          "stretch"
        ];
        const fontFamilyEntries = !disableTailwindFonts ? [
          ...extractVariables(
            patterns.FONT_FAMILY,
            themeContent
          ).filter(
            ([name]) => !invalidFontProps.some(
              (prop) => name.includes(prop)
            )
          ).map(([name, value]) => ({
            name,
            slug: name.toLowerCase(),
            fontFamily: value.replace(/['"]/g, "")
          })),
          ...resolvedTailwindConfig?.theme?.fontFamily ? processFontFamilies(
            resolvedTailwindConfig.theme.fontFamily
          ) : []
        ] : void 0;
        const fontSizeEntries = !disableTailwindFontSizes ? [
          ...extractVariables(patterns.FONT_SIZE, themeContent).filter(([name]) => !name.includes("line-height")).map(([name, value]) => ({
            name,
            slug: name.toLowerCase(),
            size: value
          })),
          ...resolvedTailwindConfig?.theme?.fontSize ? processFontSizes(
            resolvedTailwindConfig.theme.fontSize
          ) : []
        ] : void 0;
        const themeJson = {
          __processed__: "This file was generated using Vite",
          ...baseThemeJson,
          settings: {
            ...baseThemeJson.settings,
            color: disableTailwindColors ? void 0 : {
              ...baseThemeJson.settings?.color,
              palette: [
                ...baseThemeJson.settings?.color?.palette || [],
                ...colorEntries || []
              ]
            },
            typography: {
              ...baseThemeJson.settings?.typography,
              defaultFontSizes: false,
              customFontSize: false,
              fontFamilies: disableTailwindFonts ? void 0 : [
                ...baseThemeJson.settings?.typography?.fontFamilies || [],
                ...fontFamilyEntries || []
              ],
              fontSizes: disableTailwindFontSizes ? void 0 : [
                ...baseThemeJson.settings?.typography?.fontSizes || [],
                ...fontSizeEntries || []
              ]
            }
          }
        };
        delete themeJson.__preprocessed__;
        this.emitFile({
          type: "asset",
          fileName: outputPath,
          source: JSON.stringify(themeJson, null, 2)
        });
      } catch (error) {
        throw error instanceof Error ? error : new Error(String(error));
      }
    }
  };
}
export {
  wordpressPlugin,
  wordpressThemeJson
};
