<?php

namespace App\View\Components;

use Closure;
use Illuminate\Contracts\View\View;
use Illuminate\View\Component;

class Image extends Component
{
    protected $imageData = null;
    protected $webpData = null;

    /**
     * Create the component instance.
     */
    public function __construct(
        public $image,
        public ?string $class = '',
        public ?string $preset = null,
        public ?string $sizes = null,  
        public ?string $loading = 'lazy',
        public ?string $decoding = 'async'
    ) {
        // No need to reassign properties as they're already public
    }

    /**
     * Get the sizes attribute for the image
     */
    public function getSizesAttribute()
    {
        if ($this->preset) {
            return $this->getSizePreset($this->preset);
        }

        if ($this->sizes) {
            return $this->sizes;
        }
        
        return $this->getDefaultSizes();
    }

    /**
     * Get default responsive sizes optimized for mobile performance
     */
    public function getDefaultSizes()
    {
        return <<<SIZES
            (max-width: 320px) 100vw,
            (max-width: 480px) 100vw,
            (max-width: 768px) 100vw,
            (max-width: 1024px) 85vw,
            (max-width: 1280px) 75vw,
            85vw
        SIZES;
    }

    /**
     * Get optimized size presets based on image context and layout
     */
    public function getSizePreset($preset = 'default')
    {
        $presets = [
            'default' => $this->getDefaultSizes(),
            'container' => " (max-width: 768px) 100vw, 90vw ",       
            'half' => " (max-width: 768px) 100vw, 50vw ",
            'third' => " (max-width: 768px) 100vw, 33vw ",    
            'quarter' => " (max-width: 768px) 100vw, 25vw",
        ];
        
        return $presets[$preset] ?? $presets['default'];
    }

    /**
     * Determine if image is an ACF image array
     */
    protected function isAcfImage()
    {
        return isset($this->image->ID) && isset($this->image->sizes);
    }

    public function getImageData()
    {
        if (empty($this->image)) {
            return null;
        }

        // Direct ACF image object
        if ($this->isAcfImage()) {
            return $this->formatImageData(
              sizes: $this->image->sizes, 
              fullUrl: $this->image->url, 
              width: $this->image->width, 
              alt: $this->image->alt ?? ($this->image->title ?? 'Image')
            );
        }

        // ACF image as array
        if (is_array($this->image) && isset($this->image['ID'], $this->image['sizes'])) {

            return $this->formatImageData(
              sizes: $this->image['sizes'], 
              fullUrl: $this->image['url'], 
              width: $this->image['width'], 
              alt: $this->image['alt'] ?? ($this->image['title'] ?? 'Image')
            );
        }

        // Attachment ID
        $id = is_numeric($this->image) ? $this->image : 0;
        if (!$id || !wp_attachment_is_image($id)) {
            return null;
        }

        return $this->formatImageData(
            [
                'medium' => wp_get_attachment_image_url($id, 'medium'),
                'medium_2x' => wp_get_attachment_image_url($id, 'medium_large'),
                'medium_large' => wp_get_attachment_image_url($id, 'medium_large'),
                'medium_large_2x' => wp_get_attachment_image_url($id, 'large'),
                'large' => wp_get_attachment_image_url($id, 'large'),
                'large_2x' => wp_get_attachment_image_url($id, 'full'),
                'url' => wp_get_attachment_image_url($id, 'full'),
                'medium-width' => wp_get_attachment_image_src($id, 'medium')[1] ?? 300,
                'medium_large-width' => wp_get_attachment_image_src($id, 'medium_large')[1] ?? 768,
                'large-width' => wp_get_attachment_image_src($id, 'large')[1] ?? 1024,
                'medium-height' => wp_get_attachment_image_src($id, 'medium')[2] ?? '',
                'medium_large-height' => wp_get_attachment_image_src($id, 'medium_large')[2] ?? ''
            ],
            wp_get_attachment_image_url($id, 'full'),
            wp_get_attachment_metadata($id)['width'] ?? '',
            get_post_meta($id, '_wp_attachment_image_alt', true) ?: get_the_title($id) ?: 'Image'
        );
    }
    
    private function formatImageData($sizes, $fullUrl, $width, $alt)
    {
        $sizes = is_object($sizes) ? (array) $sizes : $sizes;
        return [
            'medium' => $sizes['medium'] ?? '',
            'medium_2x' => $sizes['medium_large'] ?? '',
            'medium_large' => $sizes['medium_large'] ?? '',
            'medium_large_2x' => $sizes['large'] ?? '',
            'large' => $sizes['large'] ?? '',
            'large_2x' => $fullUrl,
            'url' => $fullUrl,
            'medium_width' => $sizes['medium-width'] ?? 300,
            'medium_large_width' => $sizes['medium_large-width'] ?? 768,
            'large_width' => $sizes['large-width'] ?? 1024,
            'width' => $width,
            'medium_height' => $sizes['medium-height'] ?? '',
            'medium_large_height' => $sizes['medium_large-height'] ?? '',
            'alt' => $alt
        ];
    }
    public function getWebpData()
    {
        if (empty($this->imageData)) {
            $this->imageData = $this->getImageData();
        }

        if (!$this->imageData || empty($this->imageData['url'])) {
            return null;
        }

        $webpData = [];

        foreach (['thumbnail', 'medium_large','medium', 'large', 'url'] as $size) {
            if (!empty($this->imageData[$size])) {
                $webpUrl = preg_replace('/\.(jpg|jpeg|png)$/i', '.webp', $this->imageData[$size]);

                if ($this->imageExistsFromUrl($webpUrl)) {
                  $webpData[$size] = $webpUrl;  
                }
            }
        }

        return !empty($webpData) ? $webpData : null;
    }

    function imageExistsFromUrl($imageUrl)
    {
        // Récupérer le chemin des uploads
        $uploadDir = wp_upload_dir();
        
        // Convertir l'URL de l'image en chemin absolu
        $absolutePath = str_replace($uploadDir['baseurl'], $uploadDir['basedir'], $imageUrl);

        // Normalize le chemin pour éviter les problèmes liés aux séparateurs de dossiers
        $absolutePath = wp_normalize_path($absolutePath);

        // dump($absolutePath, file_exists($absolutePath));

        return file_exists($absolutePath);
    }

    /**
     * Get the view / contents that represent the component.
     */
    public function render(): View|Closure|string
    {
        return view('components.image');
    }
}
